// Persistence of Vision Ray Tracer Scene Description File
// File: Object Distribution_test.pov
// Vers: 3.71, 3.8
// Desc: Distribution of objects, trees, or grass on a given surface
// Date: February 2017 - June 2019
// Auth: Thomas de Groot
//

//+w640 +h640 +am2 +a0.3 +bm2 +bs8 +wt6

#version 3.8;
//#version unofficial patch 3.71; //UberPOV

#include "colors.inc"
#include "rand.inc"

global_settings {
  assumed_gamma 1.0
}

// ----------------------------------------

camera {
  location  <0.0, 15.0, -15.0>
  direction 1.5*z
  right     x*image_width/image_height
  look_at   <0.0, 2.0,  0.0>
  angle 75*image_width/image_height
}

sky_sphere {
  pigment {
    gradient y
    color_map {
      [0.0 rgb <0.6,0.7,1.0>]
      [0.7 rgb <0.0,0.1,0.8>]
    }
  }
}

light_source {
  <0, 0, 0>            // light's position (translated below)
  color rgb <1, 1, 1>  // light's color
  translate <-30, 30, -30>
}

// ----------------------------------------

plane {
  y, -1
  pigment { color rgb <0.7,0.5,0.3> }
}

#declare TestObject =
sphere {
  0.0, 1
  texture {
    pigment {
      radial
      frequency 8
      color_map {
        [0.00 color rgb <1.0,0.4,0.2> ]
        [0.33 color rgb <0.2,0.4,1.0> ]
        [0.66 color rgb <0.4,1.0,0.2> ]
        [1.00 color rgb <1.0,0.4,0.2> ]
      }
    }
    finish{
      specular 0.6
    }
  }
}

//======================================================================================================================================
// A landscape code by Kirk Andrews:

#declare GPig =
//texture {
  pigment {
    cells
    turbulence 1 lambda 10
    poly_wave 2
    scale .2
    color_map {
      [0 rgb <.1,.2,.0>]
      [.5 rgb <.1,.1,0>]
      [1 rgb <.35,.4,.1>]
    }
  }
//}

#declare GPig2 =
//texture {
  pigment {
    cells
    turbulence 1 lambda 10
    poly_wave 2
    scale .1
    color_map {
      [0 rgb <.2, .3, .0>]
      [.5 rgbt <.1, .2, 0, 1>]
      [1 rgb <.6, .5, .3>]
    }
  }
//}


#declare GTex =
texture {
  pigment {
    slope y 
    turbulence .5 lambda 20
    pigment_map {
      [0.000 rgbt <.1,.2,.0,1>]
      [0.850 rgbt <.1,.2,.0,1>]
      [0.850 GPig2]
      [1.000 GPig]
    }
  }
  normal {
    wrinkles .1
    scale .01
  }
}


#declare TTex =
texture {
  pigment {
    crackle solid 
    scale 20
    warp {turbulence .5 lambda 8}
    scale 1/20
    scale .5
    scale <1, 0.3, 1>
  }
  normal {
    //slope y
    //normal_map {
    //  [0.0 bumps .1]
    //  [0.5 granite 3 scale .25*y scale .01]
    //  [1.0 bumps .1]           
    //}
    //scale .1 
    granite scale .005
  }
}
texture {GTex}

#declare RockPig =
pigment {
  slope y
    color_map {
      [0.0 rgb <.5,.3,.2>]
      [0.5 rgb <.3,.1,.0>]
      [1.0 rgb <.7,.5,.3>]
    } 
}    

#declare RTex =
texture {
  pigment {RockPig
    //cells 
    ////turbulence .9
    //color_map {
    //  [0.0 rgb <.5,.3,.2>]
    //  [0.5 rgb <.3,.1,.0>]
    //  [1.0 rgb <.7,.5,.3>]
    //} 
    //warp {turbulence 1 lambda 10}
    //scale <6,1,6> 
    //warp {turbulence .2}
    //rotate 20*x
  }              
  normal {
    granite
    scale <1, 0.3, 1>
    scale 3
  }
}
texture {GTex rotate 5*x}

//---------------------------------------------------
#declare TS = 10; //terrain scale

#declare r = seed(3);

#declare F1 =
function {
  pattern {
    wrinkles 
    poly_wave 1/4
    scale 0.2
    translate -20
  }
}

#declare F2 =
function {
  pattern {
    wrinkles 
    poly_wave 4
    scale 0.8
    translate -150
  }
}

#declare F3 =
function {
  pattern {
    cylindrical
    scale 0.5
    translate 0.5
  }
}

#declare Res = 2000;
#declare Terrain = 
height_field {
	function Res, Res {(F1(x,y,z)*0.5 + F2(x,y,z)*2 + (1-F3(x,y,z)) )/2}

	translate <-0.5, -0.5, -0.5>
	scale 2
	scale <TS, TS/2, TS>
	  
	texture {RTex scale .1} 
}

Terrain

//======================================================================================================================================
#declare TestType = 1;  //what test to perform: 0=nothing; 1=objects; 2=trees; 3=grass

#switch (TestType)
#case (0)
#break
#case (1)

// (0) -------------------------------------------------------------------------------
// Preliminaries
#declare R       = seed(10);
#declare Target  = object {Terrain}	//fill in here the correct name
#declare MinEx   = min_extent(Target);
#declare MaxEx   = max_extent(Target);
#declare TestBox = box {<MinEx.x, 100, MinEx.z>, <MaxEx.x, 101, MaxEx.z>} // A test box where the random points will be generated
#declare Write   = on; // set off/on
#declare Num     = 600; // number of objects to trace
 
// (1) -------------------------------------------------------------------------------
// Based originally on code by Kirk Andrews (LandscapeTutorial.pov)
// ================================================================

//This is a simple Proximity Function.
//To be used for many things in a landscape scene:
#declare ProxF =
function {
  pigment {
    average
    pigment_map { 
      #local Counter = 0;
      #while (Counter < 50)
        [   
          // The object pattern, using the terrain.
          object {            
            object {Target
              translate <rand(R)-0.5, rand(R)-0.5, rand(R)-0.5>*rand(R)
            }
            color rgb 0,      // outside object
            color rgb 1       // inside object  
          } 
        ]                                                
        #local Counter = Counter+1;
      #end	//while
    }	//pigment_map
  }	//pigment
}	//function

//this influences the shape of the individual objects.
//Try different patterns:
#declare ObjF =
function {
  pattern {
    //agate
    cylindrical
    //bumps
    turbulence 0.001
    scale TS 			//scale up or down according to need
  }
}

// (2) -------------------------------------------------------------------------------
// writing random locations to file:
#declare Counter = 0; 

#if (Write)
  #debug "   writing object locations...\n"
  #fopen ObjLocs "Distribution Object.inc" write

  #while (Counter <= Num)
    // (2.1) Random location 
    #declare Norm  = <0, 0, 0>;
    #declare Start = VRand_In_Obj(TestBox, R);
    #declare Pos   = trace (Target, Start, -y, Norm);   
    
    // (2.2) Testing presence (change if necessary)                                                       
    //#declare Visible = IsObjectVisible(Pos)
    //#if (Visible)
      #if (
        //Pos.y >= 16.0
        //& Pos.y > .01*MaxY                           //  Altitude controls:  Minimum
        //&Pos.y < (.5+rand(R)*.4)*MaxY                //     "         "   :  Maximum, with a little feathering
        //vdot(Norm, y) > 0.8                          //  Limit the slope on which objects can be;  vdot is a dot product function.
        ObjF(Pos.x,Pos.y,Pos.z) < 0.5//+rand(R)*.1      //  I use a function to gather objects together, while some bare spots remain.  The rand() helps feather the edges. 
        //& ProxF(Pos.x,Pos.y,Pos.z).red > 0.45        //  Here's that proximty function again.  No objects on sharp edges.  
      )
    
        // (2.3) Writing the locations (last line test)
        #if (Counter = Num)
          #write (ObjLocs,Pos,", ",Norm,"\n")
        #else
          #write (ObjLocs,Pos,", ",Norm,",\n")
        #end

        // (2.4) Just a little code to let me know how the posing is coming:
        #if (mod(Counter, Num/10) = 0)    
          #debug concat("   Objects: ", str(Counter,5,0), "\n") 
        #end 
     
        #local Counter = Counter + 1;  // The counter only ticks if an object is actually posed. 
      #end	//of if conditions
    //#end	//of if Visible
  #end	//of while loop

  #fclose ObjLocs

#end	//of Write

// (3) -------------------------------------------------------------------------------
// Reading object locations:
#debug "   reading object locations...\n"

#declare SampleObj =
sphere { 
  <0, 0.0, 0>, 0.5
  pigment {rgb <1,0,0>}
}

#fopen ObjLocs "Distribution Object.inc" read
#declare Counter = 0;

union {
  #while (defined(ObjLocs))
    #read (ObjLocs, Pos, Norm)
    
    #declare Counter = Counter + 1;
    // Just a little code to let me know how the posing is coming:
    #if (mod(Counter, Num/10) = 0)    
      #debug concat("   Objects: ", str(Counter,5,0), "\n") 
    #end 
     
    object {
      SampleObj
           
      // The following tranformations are all about increasing the perceived variety of objects/trees in the scene:        
      //translate -rand(R)*.3*y                                             // This trick will work as long as your camera isn't too close. It adds to the perceived variety of objects.
      //scale 0.01                                                             // POV-Trees are 1 unit tall.  You'll have to figure out how tall your objects should be in your scene.
      //scale <0.5+rand(R), 0.5+ObjF(Pos.x,Pos.y,Pos.z)*2, 0.5+rand(R)>     // scale variation 1
      //scale <1, 0.5+ObjF(Pos.x,Pos.y,Pos.z)*2, 1>                         // scale variation 2
      scale 0.01 + ObjF(Pos.x,Pos.y,Pos.z)*1                              // scale variation 3
      //#if (rand(R) > .5) scale <-1, 1, 1> #end                            // Flip some objects 
      scale <1, 1+ProxF(Pos.x,Pos.y,Pos.z).red, 1>                        // objects/trees in ditches can be taller. The ProxF function should provide a decent approximation.
      //rotate rand(R)*4*x
      //rotate rand(R)*360*y
      //Reorient_Trans(y, Norm)
      translate Pos                             // and finally, pose it!
    }   
             
  #end	//of while defined

}	//end of union

#debug "Done.\n\n"

#break
#case(2)

// (1) -------------------------------------------------------------------------------
// Based on code by Kirk Andrews (LandscapeTutorial.pov)
// =====================================================
//  This is a simple Proximity Function.
//  To be used for many things in a landscape scene.
#declare r         = seed(10);
#declare Landscape = Underground;	//fill in here the correct name

#declare ProxF =
function {
  pigment {
    average
    pigment_map { 
      #local i = 0;
      #while (i < 50)
        [   
          // The object pattern, using the terrain.
          object {            
            object {Landscape
              translate <rand(r)-0.5, rand(r)-0.5, rand(r)-0.5>*rand(r)
            }
            color rgb 0,      // outside object
            color rgb 1       // inside object  
          } 
        ]                                                
        #local i = i+1;
      #end	//while
    }	//pigment_map
  }	//pigment
}	//function

#declare ObjF =
function {
  pattern {
    bumps
    turbulence 0.1
    scale 1.0 			//scale up or down according to need
  }
}

// (2) -------------------------------------------------------------------------------
// writing random locations to file:
#declare i = 0; 

#if (WriteTrees)
  #debug "   writing tree locations...\n"
  #fopen TreeLocs "TreeLocs.inc" write

  #declare TS = 300;	                      // TerrainScale (scope of the heightfield or area around the origin) 
  #declare MaxY = max_extent(Landscape).y;  // Find the highest altitude of the terrain

  #while (i <= NumTrees) 
    #declare Norm = <0, 0, 0>;
    
    // Pick a random place within the scope of the terrain
    #declare Start = <(rand(r)-0.5)*2*TS, 500, (rand(r)-0.5)*2*TS>;
    
    // Find out where that random spot is on the ground
    //                    target     spot   direction  normal
    #declare Pos = trace (Landscape, Start, -y,        Norm );   
    
    // We've got our random spot, but should an object be there?                                                       
    #declare Visible = IsObjectVisible(Pos)
    #if (Visible)
      #if (
         //Pos.y >= 16.0
         //& Pos.y > .01*MaxY                           //  Altitude controls:  Minimum
         //&Pos.y < (.5+rand(r)*.4)*MaxY                //     "         "   :  Maximum, with a little feathering
         vdot(Norm, y) > 0.8                          //  Limit the slope on which the tree grows.  vdot is a dot product function.
         & ObjF(Pos.x,Pos.y,Pos.z) < 0.5+rand(r)*.1   //  I use a function to gather trees together, while some bare spots remain.  The rand() helps feather the edges. 
         & ProxF(Pos.x,Pos.y,Pos.z).red > 0.45        //  Here's that proximty function again.  Trees shouldn't grow on the peaks of rocky ridges.  
        )
    
        //Writing the last line in the file:
        #if (i = NumTrees)
          #write (TreeLocs,Pos,", ",Norm,"\n")
        #else
          #write (TreeLocs,Pos,", ",Norm,",\n")
        #end

        // Just a little code to let me know how the posing is coming:
        #if (mod(i,NumTrees/10) = 0)    
          #debug concat("   Trees: ", str(i,5,0), "\n") 
        #end 
     
        #local i = i + 1;  // The counter only ticks if an object is actually posed. 
      #end	//of if conditions
    #end	//of if Visible
  #end	//of while loop

  #fclose TreeLocs

#end	//of Write

// (3) -------------------------------------------------------------------------------
// Reading tree locations:
#debug "   reading tree locations...\n"

//#include "DistantTree3.inc"   
  
  
// If you don't have a tree file, this'll do for now:
#declare WOOD =
cone {
  1*y,  0.0,
  -1*y, 0.1
  // open 
  texture {
    pigment {rgb <.3,.2,.1>}
  }
}

#declare FOLIAGE =
cone {
  1*y, 0.0,
  .2*y, .7
}
  
#declare TREE =
union {
  object {WOOD}
  object {FOLIAGE}
}

#fopen TreeLocs "TreeLocs.inc" read
#declare Count = 0;

union {
  #while (defined(TreeLocs))
    #read (TreeLocs, Pos, Norm)
    
    #declare Count = Count + 1;
    // Just a little code to let me know how the posing is coming:
    #if (mod(Count,NumTrees/10) = 0)    
      #debug concat("   Trees: ", str(Count,5,0), "\n") 
    #end 
     
    object {
      // POV-Tree separates the resulting tree into "WOOD" and "FOLIAGE".  I like to plant some occasional dead trees,
      // without the foliage.  
      #if (rand(r) > .9)
        WOOD
      #else
        TREE
        // If you go into the generated POV-Tree .inc file, you can delete the texture attribution on the FOLIAGE mesh,
        // allowing to add it here with some variation from tree to tree.
        texture {  
          pigment {rgb <.1 + rand(r)*.05,  .2 + rand(r)*.15, 0>}
        } 
      #end 
           
      // The following tranformations are all about increasing the perceived variety of objects/trees in the scene:        
      //translate -rand(r)*.3*y                                             // This trick will work as long as your camera isn't too close. It adds to the perceived variety of trees.
      //scale 1                                                             // POV-Trees are 1 unit tall.  You'll have to figure out how tall your trees should be in your scene.
      //scale <0.5+rand(r), 0.5+ObjF(Pos.x,Pos.y,Pos.z)*2, 0.5+rand(r)>     // scale variation 1
      //scale <1, 0.5+ObjF(Pos.x,Pos.y,Pos.z)*2, 1>                         // scale variation 2
      scale 0.3 + ObjF(Pos.x,Pos.y,Pos.z)*2                               // scale variation 3
      //#if (rand(r) > .5) scale <-1, 1, 1> #end                            // Flip some trees 
      //scale <1, 1+ProxF(Pos.x,Pos.y,Pos.z).red, 1>                        // Trees in ditches should be taller. The ProxF function should provide a decent approximation.
      //rotate rand(r)*4*x
      //rotate rand(r)*360*y
      //Reorient_Trans(y, Norm)
      translate Pos                             // and finally, pose it!
    }   
             
  #end	//of while defined

}	//end of union

#debug "Done.\n\n"

#break
#case (3)

//====================================================================================
// GRASS PLANTING
//put the following two lines somewhere in your code:
#declare GrassOn    = on;
#declare WriteGrass = on;
//====================================================================================
#if (GrassOn)
#debug "Planting grass...\n"
#include "GrassPatch_01_POV_geom.inc"
#include "GrassPatch_02_POV_geom.inc"
#include "GrassPatch_03_POV_geom.inc"
#include "GrassPatch_04_POV_geom.inc"
#include "GrassPatch_05_POV_geom.inc"
#include "GrassPatch_06_POV_geom.inc"
#include "GrassPatch_07_POV_geom.inc"
#include "GrassPatch_08_POV_geom.inc"

#declare Grass1 = object {GrassPatch_01_}
#declare Grass2 = object {GrassPatch_02_}
#declare Grass3 = object {GrassPatch_03_}
#declare Grass4 = object {GrassPatch_04_}
#declare Grass5 = object {GrassPatch_05_}
#declare Grass6 = object {GrassPatch_06_}
#declare Grass7 = object {GrassPatch_07_}
#declare Grass8 = object {GrassPatch_08_}

// An image_map to serve for the grass distribution (eval_pigment)
#declare SpottedMap = 
pigment {
  image_map {
    png "SpottedGreen_tile.png"
  } 
  rotate x*90 
  translate <-0.5,-0.5,-0.5>
  scale 1
} 

//------------------------------------------------------------------------ Start eval_pig macro ---
// Macro using the image_map SpottedGreen_tile.png :
#macro EvalPig(Map,MinScale,MaxScale,MinRot,MaxRot,Seed,FirstLine)
  #local Shade = eval_pigment(Map,<Pos.x,0,Pos.z>);
  #if (Shade.red <= 15/255)
    #declare GrassObject = 1;
    #declare Grassscale = 0.20+RRand(MinScale, MaxScale, Seed);
    #declare Grassrotate = RRand(MinRot, MaxRot, Seed);
  #end
  #if ((Shade.red > 15/255) & (Shade.red <= 20/255))
    #declare GrassObject = 2;
    #declare Grassscale = 0.19+RRand(MinScale, MaxScale, Seed);
    #declare Grassrotate = RRand(MinRot, MaxRot, Seed);
  #end
  #if ((Shade.red > 20/255) & (Shade.red <= 25/255))
    #declare GrassObject = 3;
    #declare Grassscale = 0.18+RRand(MinScale, MaxScale, Seed);
    #declare Grassrotate = RRand(MinRot, MaxRot, Seed);
  #end
  #if ((Shade.red > 25/255) & (Shade.red <= 30/255))
    #declare GrassObject = 4;
    #declare Grassscale = 0.17+RRand(MinScale, MaxScale, Seed);
    #declare Grassrotate = RRand(MinRot, MaxRot, Seed);
  #end
  #if ((Shade.red > 30/255) & (Shade.red <= 35/255))
    #declare GrassObject = 5;
    #declare Grassscale = 0.16+RRand(MinScale, MaxScale, Seed);
    #declare Grassrotate = RRand(MinRot, MaxRot, Seed);
  #end
  #if ((Shade.red > 35/255) & (Shade.red <= 40/255))
    #declare GrassObject = 6;
    #declare Grassscale = 0.15+RRand(MinScale, MaxScale, Seed);
    #declare Grassrotate = RRand(MinRot, MaxRot, Seed);
  #end
  #if ((Shade.red > 40/255) & (Shade.red <= 45/255))
    #declare GrassObject = 7;
    #declare Grassscale = 0.12+RRand(MinScale, MaxScale, Seed);
    #declare Grassrotate = RRand(MinRot, MaxRot, Seed);
  #end
  #if (Shade.red > 45/255)
    #declare GrassObject = 8;
    #declare Grassscale = 0.10+RRand(MinScale, MaxScale, Seed);
    #declare Grassrotate = RRand(MinRot, MaxRot, Seed);
  #end
  #if (FirstLine)
    #write (GrassPatches,Norm,", ",Pos,", ",GrassObject,", ",Grassscale,", ",Grassrotate)
    #declare FirstLine = off;
  #else
    #write (GrassPatches,",\n",Norm,", ",Pos,", ",GrassObject,", ",Grassscale,", ",Grassrotate)
  #end
  //#local Sred = Shade.red*255;
  //#debug concat("Shade.red: ",str(Sred,5,2)," \n")
#end 
//-------------------------------------------------------------------------- End eval_pig macro ---

//------------------------------------------------------------------------ Start planting macro ---
#macro Planting(Norm,Pos,GrassObject,Grassscale,Grassrotate,Grass1,Grass2,Grass3,Grass4,Grass5,Grass6,Grass7,Grass8)
  #local GrassTrans =
  transform {
    scale Grassscale
    rotate Grassrotate*y
    Reorient_Trans(y, Norm)
    translate Pos
  }
  
  #switch (GrassObject)
    #case (1)
      object {
        Grass1
        transform {GrassTrans}
        //material {fog_wrap (texture {Grass_01 transform {GrassTrans}}, sky_pigment) interior {ior 1.63}}
      }
    #break
    #case (2)
      object {
        Grass2
        transform {GrassTrans}
        //material {fog_wrap (texture {Grass_02 transform {GrassTrans}}, sky_pigment) interior {ior 1.63}}
      }
    #break
    #case (3)
      object {
        Grass3
        transform {GrassTrans}
        //material {fog_wrap (texture {Grass_03 transform {GrassTrans}}, sky_pigment) interior {ior 1.63}}
      }
    #break
    #case (4)
      object {
        Grass4
        transform {GrassTrans}
        //material {fog_wrap (texture {Grass_04 transform {GrassTrans}}, sky_pigment) interior {ior 1.63}}
      }
    #break
    #case (5)
      object {
        Grass5
        transform {GrassTrans}
        //material {fog_wrap (texture {Grass_05 transform {GrassTrans}}, sky_pigment) interior {ior 1.63}}
      }
    #break
    #case (6)
      object {
        Grass6
        transform {GrassTrans}
        //material {fog_wrap (texture {Grass_06 transform {GrassTrans}}, sky_pigment) interior {ior 1.63}}
      }
    #break
    #if (GrassObject = 7)
      object {
        Grass7
        transform {GrassTrans}
        //material {fog_wrap (texture {Grass_07 transform {GrassTrans}}, sky_pigment) interior {ior 1.63}}
      }
    #break
    #case (8)
      object {
        Grass8
        transform {GrassTrans}
        //material {fog_wrap (texture {Grass_08 transform {GrassTrans}}, sky_pigment) interior {ior 1.63}}
      }
    #end  //of case
  #end  //of switch
#end  //of macro
//-------------------------------------------------------------------------- End planting macro ---

// Planting the grass patches:
#declare Seed    = seed(1);
#declare Spacing =  0.1;   //distance between patches
#declare PosX    = -5.0;   //starting x location
#declare EndX    =  5.0;   //ending x location
#declare IniZ    = -5.0;  //starting z location
#declare EndZ    =  5.0;  //ending z location

#declare GrassField =
object {MySurface}  //this is the landscape on which the grass is traced.

#if (WriteGrass)
  #debug "   Writing grass patch positions...\n"
  #fopen GrassPatches "GrassPatches.inc" write
  #declare FirstLine = on;
  #declare Count     = 0;
  #declare Counter   = 100;

  #while (PosX < EndX)
    #declare PosZ = IniZ;

    #while (PosZ < EndZ)
      // trace function
      #declare Norm = <0, 0, 0>;
      #declare Start = <PosX + RRand(-Spacing/2, Spacing*2, Seed), 10, PosZ + RRand(-Spacing/2, Spacing*2, Seed)>;
      #declare Pos = trace (
                     GrassField, // object to test
                     Start,      // starting point
                     -y,         // direction
                     Norm );     // normal


      #declare Visible = IsObjectVisible(Pos)
      #if (vlength(Norm) & Visible) //add a height or other condition if necessary
        EvalPig(SpottedMap, 0.0, 0.5, -180, 180, Seed,FirstLine)  // writes selection of objects according to eval_pigment of the image_map
        #declare Count = Count + 1;
        #if (Count = Counter)
          #debug concat("     Patches written: ", str(Count,5,0), "\n")
          #declare Counter = Counter + 100;
        #end
      #end //end of Norm, minimum position, and visibility test

      #declare PosZ = PosZ + RRand(Spacing/4,Spacing*2,Seed);
    #end //end of while Z

    #declare PosX = PosX + RRand(Spacing/4,Spacing*2,Seed);
  #end //end of while X

  #fclose GrassPatches
  #debug "   Done.\n\n"
#end	//of WriteGrass

//-------------------------------------------------------------------------- Reading locations ---
#debug "   Reading grass patch positions and adding flowers...\n"
#fopen GrassPatches "GrassPatches.inc" read
#include "PlantStudio files\Wild_Pink_POV_geom.inc"
#include "PlantStudio files\Buttercup_POV_geom.inc"

#declare FlowerWP     = object {Wild_Pink_ scale 0.25*0.001}
#declare FlowerBC     = object {Buttercup_ translate <650, 470, 0> scale 0.15*0.00025}
#declare Probability1 = 0.2;
#declare Probability2 = 0.8;
#declare FProb        = seed(2);
#declare Count        = 0;
#declare Counter      = 100;

#while (defined(GrassPatches))

  #declare Count = Count + 1;
  #if (Count = Counter)
    #debug concat("     Patches read: ", str(Count,5,0), "\n")
    #declare Counter = Counter + 100;
  #end

  #read (GrassPatches, Norm, Pos, GrassObject, Grassscale, Grassrotate)
  Planting(Norm,Pos,GrassObject,Grassscale,Grassrotate,Grass1,Grass2,Grass3,Grass4,Grass5,Grass6,Grass7,Grass8)	//plants the grass patches
  
  #declare Flowers = Prob(Probability1, FProb);
  #if (Flowers)
    #declare N = RRand(1,5,FProb);
    #while (N >= 0)
      //#if (!inside(Object01,Pos) | !inside(Object02, Pos))  //use this to exclude objects on the grass. Fill in correct object name.
        #local Ftype = Prob(Probability2, FProb);
        #if (Ftype)
          object {FlowerBC 
            scale RRand(1.0, 1.50, FProb) 
            rotate <RRand(-1,1,FProb), RRand(-180,180,FProb), RRand(-1,1,FProb)> 
            translate Pos + <RRand(-0.01,0.01,FProb), 0, RRand(-0.01,0.01,FProb)>
          }
        #else
          object {FlowerWP 
            scale RRand(1.0, 1.50, FProb) 
            rotate <RRand(-1,1,FProb), RRand(-180,180,FProb), RRand(-1,1,FProb)> 
            translate Pos + <RRand(-0.1,0.1,FProb), 0, RRand(-0.1,0.1,FProb)>
          }
        #end	//of Ftype
      //#end	//of inside()
      #declare N = N - 1;
    #end	//of N
  #end	//of Flowers
  
#end	//of defined()

#debug "   Done.\n"
#debug "Grass planting done.\n\n"

#end	//of GrassOn

#end  //of TestType switch
